<?php if (!defined('ROOTPATH')) exit('No direct script access allowed'); ?>
<?php

/**
 * Redmine Defect Plugin for TestRail
 *
 * Copyright Gurock Software GmbH. All rights reserved.
 *
 * This is the TestRail defect plugin for Redmine. Please see
 * http://docs.gurock.com/testrail-integration/defects-plugins for
 * more information about TestRail's defect plugins.
 *
 * http://www.gurock.com/testrail/
 */
class Redmine_customfield_defect_plugin extends Defect_plugin
{
	private $_api;
	
	private $_address;
	private $_user;
	private $_password;
	private $_token;

	private $_is_legacy = false;
	private $_is_basic_auth = false;
	private $_trackers;
	private $_categories;
	private $_custom_fields = [];

	const msg_redmine_custom_project = 'プロジェクト';
	const msg_redmine_custom_description = '概要';
	const msg_redmine_custom_category = 'カテゴリー';
	const msg_redmine_custom_assigned = '担当者';
	const msg_redmine_custom_subject = '題目';
	const msg_redmine_custom_status = 'ステータス';
	const msg_redmine_custom_parent = '親チケット';
	const msg_redmine_custom_estimated_hours = '予定工数';
	const msg_redmine_custom_tracker = 'トラッカー';
	const msg_redmine_custom_yes = 'はい';
	const msg_redmine_custom_no = 'いいえ';

	private static $_meta = array(
		'author' => 'TechMatrix',
		'version' => '1.5',
		'description' => 'Redmine defect plugin for TestRail with custom field',
		'can_push' => true,
		'can_lookup' => true,
		'default_config' =>
		'; Redmineの1.3以降のバージョンを指定してください。
; 認証方法およびアカウント/APIトークンの指定
;
; user/passwordの認証を使う→ userとpasswordを指定してください。
; API tokenを使う→ tokenを指定してください。
;
[connection]
address=http://<your-server>/
user=%redmine_user%
password=%redmine_password%
token=api_token

; カスタムフィールドの指定
;
; json形式で指定してください。
; fields=[{"type":<type>, "label":<label>, "fid":<fid>, "required":"<true|false>", "multi":"<true|false>"}, {}]
; 例）
; fields = [{"fid":"3", "type":"string", "label":"コメント", "required":"false"}, {"fid":"4", "type":"string", "label":"作業難易度", "required":"false"}, {"fid":"5", "type":"string", "label":"ポイント", "required":"false"}, {"fid":"6", "type":"keyvalue", "label":"工程", "required":"false"}]
;
;  type: フィールドタイプ [string, text, list, keyvalue, bool, date, user]
;  label: チケット登録/参照時の表示名
;  fid: カスタムフィールドのID
;  required: 必須入力 [true, false]
;  multi: 複数選択 [true, false] ※list, keyvalueのみ。省略した場合はfalse
;
[custom_field]
fields= [{"type":"string",  "label": "カスタムフィールド", "fid": 9, "required": "false" }]
	');
	
	public function get_meta()
	{
		return self::$_meta;
	}
	
	// *********************************************************
	// CONFIGURATION
	// *********************************************************
	
	public function validate_config($config)
	{
		$ini = ini::parse($config);
		
		if (!isset($ini['connection']))
		{
			throw new ValidationException('Missing [connection] group');
		}
		
		$keys = array('address');
		
		// Check required values for existance
		foreach ($keys as $key)
		{
			if (!isset($ini['connection'][$key]) ||
				!$ini['connection'][$key])
			{
				throw new ValidationException(
					"Missing configuration for key '$key'"
				);
			}
		}
		
		$address = $ini['connection']['address'];
		
		// Check whether the address is a valid url (syntax only)
		if (!check::url($address))
		{
			throw new ValidationException('Address is not a valid url');
		}

		if (isset($ini['connection']['mode']))
		{
			// Mode must be set to 'legacy' when available.
			if ($ini['connection']['mode'] != 'legacy')
			{
				throw new ValidationException(
					'Mode given but not set to "legacy"'
				);
			}

			if (!isset($ini['trackers']))
			{
				throw new ValidationException(
					'Using legacy mode but [trackers] is missing'
				);
			}

			if (!isset($ini['categories']))
			{
				throw new ValidationException(
					'Using legacy mode but [categories] is missing'
				);
			}
		}
	}
	
	public function configure($config)
	{
		$ini = ini::parse($config);
		$this->_address = str::slash($ini['connection']['address']);

		if (isset($ini['connection']['user']))
		{
			$this->_user = $ini['connection']['user'];
		}
		if (isset($ini['connection']['password']))
		{
			$this->_password = $ini['connection']['password'];
			$this->_is_basic_auth = true;
		}
		if (isset($ini['connection']['token']))
		{
			$this->_token = $ini['connection']['token'];
			$this->_is_basic_auth = false;
		}
		if (isset($ini['connection']['mode']))
		{
			$this->_is_legacy = true;
			$this->_trackers = $ini['trackers'];
			$this->_categories = $this->_parse_categories(
				$ini['categories']);
		}
		if (isset($ini['custom_field']))
		{
			$this->_custom_fields = json_decode($ini['custom_field']['fields'], true);
			# 内部的にフィールドタイプを変更
			for ($i = 0; $i < count($this->_custom_fields); $i++) {
				if(isset($this->_custom_fields[$i]['multi'])){
					# list && multi
					if (($this->_custom_fields[$i]['type'] == "list") && ($this->_custom_fields[$i]['multi'] == "true")) {
						$this->_custom_fields[$i]['type'] = "listmulti";
					# keyvalue && multi
					} else if (($this->_custom_fields[$i]['type'] == "keyvalue") && ($this->_custom_fields[$i]['multi'] == "true")) {
						$this->_custom_fields[$i]['type'] = "keyvaluemulti";
					}
				}
			}
		}
	}
	
	private function _parse_categories($ini)
	{
		$categories = array();

		// Uses the given ini section with keys 'project_id.item_id'
		// to create a category key => value mapping for the given
		// projects.
		foreach ($ini as $key => $value)
		{
			if (preg_match('/^([^\.]+)\.([^\.]+)$/', $key, $matches))
			{
				$project_id = (int) $matches[1];
				$item_id = (int) $matches[2];
				$categories[$project_id][$item_id] = $value;
			}
		}

		return $categories;
	}

	// *********************************************************
	// API / CONNECTION
	// *********************************************************
	
	private function _get_api()
	{
		if ($this->_api)
		{
			return $this->_api;
		}
		
		$this->_api = new Redmine_custom_api(
			$this->_address,
			$this->_user,
			$this->_password,
			$this->_token,
		    $this->_is_basic_auth);
		
		return $this->_api;
	}
	
	// *********************************************************
	// PUSH
	// *********************************************************

	public function prepare_push($context)
	{
		// Return a form with the following fields/properties
		$default_array =  array(
			'fields' => array(
				'subject' => array(
					'type' => 'string',
					'label' => self::msg_redmine_custom_subject,
					'required' => true,
					'size' => 'full'
				),
				'tracker' => array(
					'type' => 'dropdown',
					'label' => self::msg_redmine_custom_tracker,
					'required' => true,
					'remember' => true,
					'size' => 'compact'
				),
				'project' => array(
					'type' => 'dropdown',
					'label' => self::msg_redmine_custom_project,
					'required' => true,
					'remember' => true,
					'cascading' => true,
					'size' => 'compact'
				),
				'category' => array(
					'type' => 'dropdown',
					'label' => self::msg_redmine_custom_category,
					'remember' => true,
					'depends_on' => 'project',
					'size' => 'compact'
				),
				'assigned_to' => array(
					'type' => 'dropdown',
					'label' => self::msg_redmine_custom_assigned,
					'remember' => true,
					'depends_on' => 'project',
					'size' => 'compact'
				),
				'parent' => array(
					'type' => 'string',
					'label' => self::msg_redmine_custom_parent,
					'remember' => true,
					'size' => 'compact'
				),
				'estimated_hours' => array(
					'type' => 'string',
					'label' => self::msg_redmine_custom_estimated_hours,
					'remember' => true,
					'size' => 'compact'
				),
				'description' => array(
					'type' => 'text',
					'label' => self::msg_redmine_custom_description,
					'rows' => 10
				)
			)
		);

		if (count($this->_custom_fields) > 0){
		
			foreach ($this->_custom_fields as $field) {
				$tmp_array = array();
				$tmp_array['type'] = 'text';
				$tmp_array['size'] = 'compact';
				switch($field['type']){
				case 'list':
					$tmp_array['type'] = 'dropdown';
					$tmp_array['size'] = 'compact';
					break;
				case 'bool':
					$tmp_array['type'] = 'dropdown';
					$tmp_array['size'] = 'compact';
					break;
				case 'string':
					$tmp_array['type'] = 'string';
					$tmp_array['size'] = 'compact';
					break;
				case 'text':
					$tmp_array['type'] = 'text';
					$tmp_array['size'] = 'full';
					break;
				case 'listmulti':
					$tmp_array['type'] = 'multiselect';
					$tmp_array['size'] = 'compact';
					break;
				case 'date':
					$tmp_array['type'] = 'string';
					$tmp_array['size'] = 'compact';
					break;
				case 'user':
					$tmp_array['type'] = 'dropdown';
					$tmp_array['size'] = 'compact';
					$tmp_array['depends_on'] = 'project';
					break;
				case 'keyvalue':
					$tmp_array['type'] = 'dropdown';
					$tmp_array['size'] = 'compact';
					break;
				case 'keyvaluemulti':
					$tmp_array['type'] = 'multiselect';
					$tmp_array['size'] = 'compact';
					break;
				}
				$tmp_array['label'] = $field['label'];
				if($field['required'] == 'true'){
					$tmp_array['required'] = 'true';
				}
				$default_array['fields']['custom_field_'.$field['type'].'_'.$field['fid']] = $tmp_array;
			}
		}
		return $default_array;
	}
	
	private function _get_subject_default($context)
	{		
		$test = current($context['tests']);
		$subject = 'Failed test: ' . $test->case->title;
		
		if ($context['test_count'] > 1)
		{
			$subject .= ' (+others)';
		}
		
		return $subject;
	}
	
	private function _get_description_default($context)
	{
		return $context['test_change']->description;
	}
	
	private function _to_id_name_lookup($items)
	{
		$result = array();
		foreach ($items as $item)
		{
			$result[$item->id] = $item->name;
		}
		return $result;
	}

	private function _to_id_project_name_lookup($items)
	{
		$result = array();
		foreach ($items as $item)
		{
			$result[$item->project->id] = $item->project->name;
		}
		return $result;
	}
	
	private function _to_id_name_memberships_lookup($items)
	{
		$result = array();
		foreach ($items as $item)
		{
			if(array_key_exists('user', (array)$item)){ // 非メンバーなどの場合、'user'が存在しない
				$result[$item->user->id] = $item->user->name;
			}
		}
		asort($result);
		return $result;
	}

	private function _get_trackers($api)
	{
		// In legacy mode for Redmine versions older than 1.3, we use
		// the user-configured values for the trackers. Otherwise,
		// we can just use the API.
		if ($this->_is_legacy)
		{
			if (is_array($this->_trackers))
			{						
				return $this->_trackers;
			}
			else 
			{
				return null;
			}
		}
		else 
		{
			return $this->_to_id_name_lookup(
				$api->get_trackers()
			);
		}
	}

	private function _get_categories($api, $project_id)
	{
		// In legacy mode for Redmine versions older than 1.3, we use
		// the user-configured values for the categories. Otherwise,
		// we can just use the API.
		if ($this->_is_legacy)
		{
			$categories = arr::get($this->_categories, $project_id);

			if (!is_array($categories))
			{
				return null;
			}

			return $categories;
		}
		else
		{
			return $this->_to_id_name_lookup(
				$api->get_categories($project_id)
			);
		}
	}

	private function _get_projects_with_current_user($api)
	{
		return $api->get_projects_with_current_user();	
	}	

	private function _get_users($api)
	{
		return $api->get_users();	
	}	

	private function _get_memberships($api, $project_id)
	{
		return $api->get_memberships($project_id);	
	}
	
	private function _get_custom_fields($api, $id)
	{
		return $api->get_custom_fields($id);	
	}

	public function prepare_field($context, $input, $field)
	{
		$data = array();

		// Process those fields that do not need a connection to the
		// Redmine installation.		
		if ($field == 'subject' || $field == 'description')
		{
			switch ($field)
			{
				case 'subject':
					$data['default'] = $this->_get_subject_default(
						$context);
					break;
					
				case 'description':
					$data['default'] = $this->_get_description_default(
						$context);
					break;				
			}
		
			return $data;
		}
		
		// Take into account the preferences of the user, but only
		// for the initial form rendering (not for dynamic loads).
		if ($context['event'] == 'prepare')
		{
			$prefs = arr::get($context, 'preferences');
		}
		else
		{
			$prefs = null;
		}
		
		// And then try to connect/login (in case we haven't set up a
		// working connection previously in this request) and process
		// the remaining fields.
		$api = $this->_get_api();

		switch ($field)
		{
			case 'tracker':
				$data['default'] = arr::get($prefs, 'tracker');
				$data['options'] = $this->_get_trackers($api);
				break;

			case 'project':
				$data['default'] = arr::get($prefs, 'project');
				$data['options'] = $this->_get_projects_with_current_user($api);
				break;

			case 'category':
				if (isset($input['project']))
				{
					$data['default'] = arr::get($prefs, 'category');
					$data['options'] = $this->_get_categories($api,	$input['project']);
				}
				break;

			case 'assigned_to':
				if (isset($input['project']))
				{
					$data['default'] = arr::get($prefs, 'assigned_to');
					$data['options'] = $this->_to_id_name_memberships_lookup($this->_get_memberships($api, $input['project']));
				}
				break;

			case 'parent':
				$data['default'] = arr::get($prefs, 'parent');
				break;

			case 'estimated_hours':
				$data['default'] = arr::get($prefs, 'estimated_hours');
				break;

		}

		if (!empty($data)){
			return $data;
		}

		if (count($this->_custom_fields) > 0) {
			foreach ($this->_custom_fields as $elem) {
				$field_name = 'custom_field_'.$elem['type'].'_'.$elem['fid'];
				if($field_name == $field) {
					if(($elem['type'] == 'list')||($elem['type'] == 'bool')||($elem['type'] == 'listmulti') || ($elem['type'] == 'keyvalue') || ($elem['type'] == 'keyvaluemulti')) {
						$data['options'] = $this->_get_custom_fields($api, $elem['fid']);
					}
					elseif($elem['type'] == 'date'){
						$data['default'] = date("Y-m-d");
					}
					elseif($elem['type'] == 'user'){
						if (isset($input['project']))
						{
							$data['options'] = $this->_to_id_name_memberships_lookup($this->_get_memberships($api, $input['project']));
						}
					}
				}
			}
		}

		return $data;
	}
	
	public function validate_push($context, $input)
	{
	}

	public function push($context, $input)
	{
		$api = $this->_get_api();
		return $api->add_issue($input);
	}
	
	// *********************************************************
	// LOOKUP
	// *********************************************************
	
	public function lookup($defect_id)
	{
		$api = $this->_get_api();
		$issue = $api->get_issue($defect_id);

		$status_id = GI_DEFECTS_STATUS_OPEN;
		
		if (isset($issue->status))
		{
			$status = $issue->status->name;
			
			// Redmine's status API is only available in Redmine 1.3
			// or later, unfortunately, so we can only try to guess
			// by its name.
			switch (str::to_lower($status))
			{
				case 'resolved':
					$status_id = GI_DEFECTS_STATUS_RESOLVED;
					break;

				case 'closed':
					$status_id = GI_DEFECTS_STATUS_CLOSED;
					break;
			}
		}
		else 
		{
			$status = null;
		}
		
		if (isset($issue->description) && $issue->description)
		{
			$description = str::format(
				'<div class="monospace">{0}</div>',
				nl2br(
					html::link_urls(
						h($issue->description)
					)
				)
			);
		}
		else
		{
			$description = null;
		}
		
		// Add some important attributes for the issue such as the
		// current status and project.
		
		$attributes = array();

		if (isset($issue->tracker))
		{
			$attributes[self::msg_redmine_custom_tracker] = h($issue->tracker->name);
 		}

		if ($status)
		{
			$attributes[self::msg_redmine_custom_status] = h($status);
		}

		if (isset($issue->project))
		{
			// Add a link back to the project (issue list).
			$attributes[self::msg_redmine_custom_project] = str::format(
				'<a target="_blank" href="{0}projects/{1}">{2}</a>',
				a($this->_address),
				a($issue->project->id),
				h($issue->project->name)
			);
		}

		if (isset($issue->category))
		{
			$attributes[self::msg_redmine_custom_category] = h($issue->category->name);
		}

		if (isset($issue->assigned_to))
		{
			$attributes[self::msg_redmine_custom_assigned] = h($issue->assigned_to->name);
		}
		 
		if (isset($issue->parent))
		{
			$attributes[self::msg_redmine_custom_parent] = h($issue->parent);
		}

		if (isset($issue->estimated_hours))
		{
			$attributes[self::msg_redmine_custom_estimated_hours] = h($issue->estimated_hours);
		}

		if(count($this->_custom_fields) > 0){
			if (isset($issue->custom_fields))
			{
				$tmp_issues = json_decode(json_encode($custom_field_issues=$issue->custom_fields), true);

				foreach($this->_custom_fields as $field){
					foreach($tmp_issues as $vals){
						if(!empty($vals['value'])){
							if(strcmp($field['fid'], $vals['id']) == 0){
								if(strcmp($field['type'], "listmulti") == 0){
									$tmp_value = "";
									foreach($vals['value'] as $one_value){							
										if(strcmp($tmp_value, "") == 0){
											$tmp_value .= $one_value;
										}else{
											$tmp_value .= " , ".$one_value;
										}
									}
									$attributes[$field['label']] = h($tmp_value);
								}else if(strcmp($field['type'], "bool") == 0){
									if(strcmp($vals['value'], "") == 0){
										$attributes[$field['label']] = h("");
									}else if(strcmp($vals['value'], "0") == 0){
										$attributes[$field['label']] = h(self::msg_redmine_custom_no);
									}else{
										$attributes[$field['label']] = h(self::msg_redmine_custom_yes);
									}
								}else if (strcmp($field['type'], "user") == 0) {
									##カスタムフィールドのUserはnameが入っていないので、プロジェクトのメンバーリストから選択する。
									$users = $api->get_memberships($issue->project->id);
									foreach ($users as $elem) {
										if($elem->user->id == $vals['value']){
											$attributes[$field['label']] = h($elem->user->name);
										}
									}
								}else if(strcmp($field['type'], "keyvalue") == 0){
									$custom_field_options = $api->get_custom_fields($vals['id']);
									$attributes[$field['label']] = h($custom_field_options[$vals['value']]); # keyvalue is started from '1'.
								} else if (strcmp($field['type'], "keyvaluemulti") == 0) {
									$tmp_value = "";
									$custom_field_options = $api->get_custom_fields($vals['id']);
									foreach ($vals['value'] as $one_value) {
										if (strcmp($tmp_value, "") == 0) {
											$tmp_value .= $custom_field_options[$one_value];
										} else {
											$tmp_value .= " , " . $custom_field_options[$one_value];
										}
									}
									$attributes[$field['label']] = h($tmp_value);
								}else{
									$attributes[$field['label']] = h($vals['value']);
								}
								break;
							}
						}
					}
				}
			}
		}

		$default_lookup_array = array(
			'id' => $defect_id,
			'url' => str::format(
				'{0}issues/{1}',
				$this->_address,
				$defect_id
			),
			'title' => $issue->subject,
			'status_id' => $status_id,
			'status' => $status,
			'description' => $description,
			'attributes' => $attributes
		);

		return $default_lookup_array;

	}
}

/**
 * Redmine API
 *
 * Wrapper class for the Redmine API with functions for retrieving
 * projects, bugs etc. from a Redmine installation.
 */
class Redmine_custom_api
{
	private $_address;
	private $_user;
	private $_password;
	private $_token;
	private $_is_basic_auth;
	private $_version;
	private $_curl;
	private $_custom_fields_raw;
	const msg_redmine_custom_yes = 'はい';
	const msg_redmine_custom_no = 'いいえ';
	/**
	 * Construct
	 *
	 * Initializes a new Redmine API object. Expects the web address
	 * of the Redmine installation including http or https prefix.
	 */	
	public function __construct($address, $user, $password, $token, $is_basic_auth)
	{
		$this->_address = str::slash($address);
		$this->_user = $user;
		$this->_password = $password;
		$this->_token = $token;
		$this->_is_basic_auth = $is_basic_auth;
		$this->_custom_fields_raw = "";
	}
	
	private function _throw_error($format, $params = null)
	{
		$args = func_get_args();
		$format = array_shift($args);
		
		if (count($args) > 0)
		{
			$message = str::formatv($format, $args);
		}
		else 
		{
			$message = $format;
		}
		
		throw new RedmineCustomException($message);
	}
	
	private function _send_command($method, $command, $data = array())
	{
		$url = $this->_address . $command . '.json';
		$offset = 0;
		$limit = 100;
		$param = array();
		if ($method == 'GET')
		{
			array_push($param, 'limit=' . $limit);
			if (strpos($command, 'users/') !== false)
			{
				array_push($param, 'include=memberships');
			}
		}
		if (!$this->_is_basic_auth)
		{
			array_push($param, 'key=' . $this->_token);
		}
		if (!empty($param))
		{
			$url .= '?' . implode('&', $param);
		}

		// Redmine REST API used in this script.
		//  [GET]
		//   issues/$issueid
		//   trackers
		//   projects
		//   users/current
		//   projects/$project_id/issue_categories
		//   projects/$project_id/memberships
		//   custom_fields
		//
		if ($method == 'GET')
		{
			$output = [];
			do
			{
				$getUrl = $url . '&offset=' . $offset;
				$newDataSet = (array)$this->_send_request($method, $getUrl, $data);
				if ((!array_key_exists('total_count', $newDataSet))||(!array_key_exists('offset', $newDataSet)))
				{
					return (object)$newDataSet;
				}
				$output = array_merge_recursive($output, $newDataSet);
				$offset = $newDataSet['offset'] + $newDataSet['limit'];
			}while ($offset < $newDataSet['total_count']);
			return (object)$output;
		}
		// Redmine REST API used in this script.
		//  [POST]
		//   issues
		elseif ($method == 'POST')
		{
			return $this->_send_request($method, $url, $data);
		}
	}
	
	private function _send_request($method, $url, $data)
	{
		if (!$this->_curl)
		{
			// Initialize the cURL handle. We re-use this handle to
			// make use of Keep-Alive, if possible.
			$this->_curl = http::open();
		}

		if (!$this->_is_basic_auth)
		{
			$response = http::request_ex(
				$this->_curl,
				$method, 
				$url, 
				array(
					'data' => $data,
					'headers' => array(
						'Content-Type' => 'application/json'
					)
				)
			);
		}
		else{
			$response = http::request_ex(
				$this->_curl,
				$method, 
				$url, 
				array(
					'data' => $data,
					'user' => $this->_user,
					'password' => $this->_password,
					'headers' => array(
						'Content-Type' => 'application/json'
					)
				)
			);
		}

		// In case debug logging is enabled, we append the data
		// we've sent and the entire request/response to the log.
		if (logger::is_on(GI_LOG_LEVEL_DEBUG))
		{
			logger::debugr('$data', $data);
			logger::debugr('$response', $response);
		}
		
		$obj = json::decode($response->content);
		
		if ($response->code != 200)
		{
			if ($response->code != 201) // Created
			{
				if(!empty($obj->errors)){
					$this->_throw_error(
						'Invalid HTTP code ({0}). '.$obj->errors[0],
						$response->code
					);
				}
				else{
					$this->_throw_error(
						'Invalid HTTP code ({0}). Please check your user/' .
						'password/token and that the API is enabled in Redmine.',
						$response->code
					);
				}
			}
		}
		return $obj;
	}

	/**
	 * Get Issue
	 *
	 * Gets an existing issue from the Redmine installation and
	 * returns it. The resulting issue object has various properties
	 * such as the subject, description, project etc.
	 */	 
	public function get_issue($issue_id)
	{
		$response = $this->_send_command(
			'GET', 'issues/' . urlencode($issue_id)
		);
		
		return $response->issue;
	}
	
	/**
	 * Get Trackers
	 *
	 * Gets the available trackers for the Redmine installation.
	 * Trackers are returned as array of objects, each with its ID
	 * and name. Requires Redmine >= 1.3.
	 */
	public function get_trackers()
	{
		$response = $this->_send_command('GET', 'trackers');
		return $response->trackers;
	}

	/**
	 * Get Projects
	 *
	 * Gets the available projects for the Redmine installation.
	 * Projects are returned as array of objects, each with its ID
	 * and name.	 
	 */
	public function get_projects()
	{
		$response = $this->_send_command('GET', 'projects');
		return $response->projects;
	}

	/**
	 * Get Projects with current user.
	 */
	public function get_projects_with_current_user()
	{
		$projects = $this->get_projects();
		$response = $this->_send_command('GET', "users/current");
		$users_raw = json_decode(json_encode($response->user->memberships), true);
		$projects_raw = json_decode(json_encode($projects), true);

		$activeIds = array();
		foreach ($projects_raw as $elem){
			if($elem['status'] == 1){
				array_push($activeIds, $elem['id']);
			}
		}

		$parray = array();
		foreach ($users_raw as $value){
			$project = $value['project'];
			foreach ($activeIds as $activeId){
				if($project['id'] == $activeId){
					$parray[$project['id']] = $project['name'];
					continue;
				}
			}
		}
		asort($parray);
		return $parray;
	}
	
	/**
	 * Get Categories
	 *
	 * Gets the available categories for the given project ID for the
	 * Redmine installation. Categories are returned as array of
	 * objects, each with its ID and name. Requires Redmine >= 1.3.
	 */
	public function get_categories($project_id)
	{
		try{
			$response = $this->_send_command('GET', 
				"projects/$project_id/issue_categories");
		}catch(Exception $ex){
			if(strpos($ex->getMessage(), '403') == false){ 
				Throw $ex;
			}
			else{ // 403 = Cannot get issue_categories, but ignore this ex.
				return array();
			}
		}
		return $response->issue_categories;
	}

	public function get_memberships($project_id)
	{
		$response = $this->_send_command('GET', 
			"projects/$project_id/memberships");
		return $response->memberships;
	}

	public function get_custom_fields($id)
	{
		$sourceUrl = parse_url($this->_address);
		$json_filename = LOG_PATH.$sourceUrl['host'].'_Redmine_custom_field.json';
		if(empty($this->_custom_field_raw)){
			try{
				$response = $this->_send_command('GET', 
					"custom_fields");
				$this->_custom_field_raw = json_decode(json_encode($response->custom_fields), true);
				$fp = fopen($json_filename, "w");
				if (flock($fp, LOCK_EX)) {
					ftruncate($fp, 0);
					fwrite($fp, json_encode($response->custom_fields));
					fflush($fp);
					flock($fp, LOCK_UN);
				}
			}catch(Exception $ex){
				if(strpos($ex->getMessage(), '403') !== false){ // 403 = Cannot get custom_fields
					$this->_custom_field_raw = json_decode(file_get_contents($json_filename), true);
				}
				else{
					Throw $ex;
				}
			}
		}

		foreach ($this->_custom_field_raw as $value){
			if($id == $value['id']){
				if($value['field_format'] == 'bool'){
					return [self::msg_redmine_custom_yes, self::msg_redmine_custom_no];
				}
				elseif($value['field_format'] == 'list'){
					$parray = [];
					foreach ($value['possible_values'] as $pvalue){
						array_push($parray, $pvalue['value']);
					}
					return $parray;
				}
				elseif($value['field_format'] == 'enumeration'){ # keyvalue
					$parray = [];
					foreach ($value['possible_values'] as $pvalue){
						$parray[$pvalue['value']] = $pvalue['label'];
					}
					return $parray;
				}
				else{
					return;
				}
			}
		}
	}

	/**
	 * Add Issue
	 *
	 * Adds a new issue to the Redmine installation with the given
	 * parameters (subject, project etc.) and returns its ID.
	 *
	 * subject:     The title of the new issue
	 * tracker:     The ID of the tracker of the new issue (bug,
	 *              feature request etc.)
	 * project:     The ID of the project the issue should be added
	 *              to
	 * category:    The ID of the category the issue is added to
	 * description: The description of the new issue
	 */	
	public function add_issue($options)
	{
		$issue = obj::create();
		$issue->subject = $options['subject'];
		$issue->description = $options['description'];
		$issue->tracker_id = $options['tracker'];
		$issue->project_id = $options['project'];
		$issue->parent_id = $options['parent'];
		$issue->estimated_hours = $options['estimated_hours'];
		
		if ($options['category'])
		{
			$issue->category_id = $options['category'];
		}

		if ($options['assigned_to'])
		{
			$issue->assigned_to_id = $options['assigned_to'];
		}

		$custom_fields_array = array();
		foreach (array_keys($options) as $key){

			if(strpos($key, 'custom_field_') !== false){
				$value = '';
				$tmp = explode('_', $key);
				$custom_field_id = str_replace('custom_field_', '', $tmp[3]);

				$custom_field_type = $tmp[2];
				if($custom_field_type == 'list'){
					$custom_field_options = $this->get_custom_fields($custom_field_id);
					if(isset($options[$key])){
						$value = $custom_field_options[$options[$key]];
					}
				}
				elseif($custom_field_type == 'listmulti'){
					if(isset($options[$key])){
						$custom_field_options = $this->get_custom_fields($custom_field_id);
						$multi_array = array();
						foreach ($options[$key] as $elem){
							array_push($multi_array, $custom_field_options[$elem]);
						}
						$value = $multi_array;
					}
				}
				elseif($custom_field_type == 'bool'){
					if(($options[$key] == 0)&&(!is_null($options[$key]))){
						$value = 1;
					}
					else{
						$value = 0;
					}
				}
				elseif (($custom_field_type == 'keyvalue')||($custom_field_type == 'keyvaluemulti')) {
					if (isset($options[$key])) {
						$value = $options[$key];
					}
				}
				else{
					$value = $options[$key];
				}
				array_push($custom_fields_array, array("id" => $custom_field_id ,"value" => $value));
			}
		}
		if(!empty($custom_fields_array)){
			logger::debugr('$custom_fields_array', $custom_fields_array);
			$issue->custom_fields = $custom_fields_array;
		}
		$data = json::encode(array('issue' => $issue));
		$response = $this->_send_command('POST', 'issues', $data);
		return $response->issue->id;
	}
}

class RedmineCustomException extends Exception
{
}
