<?php if (!defined('ROOTPATH')) exit('No direct script access allowed'); ?>
<?php

/**
 * Milestone Summary report for TestRail
 *
 * Copyright Gurock Software GmbH. All rights reserved.
 *
 * This is the TestRail report for displaying a summary/overview for a
 * milestone.
 *
 * http://www.gurock.com/testrail/
 */

class Milestones_summary_with_reliability_growth_curve_report_plugin extends Report_plugin
{
	private $_model;
	private $_controls;

	// The controls and options for those controls that are used on
	// the form of this report.
	private static $_control_schema = array(
		'milestones_select' => array(
			'namespace' => 'custom_milestones',
		),
		'activities_daterange' => array(
			'type' => 'dateranges_select',
			'namespace' => 'custom_activities'
		),
		'activities_statuses' => array(
			'type' => 'statuses_select',
			'namespace' => 'custom_activities_statuses',
		),
		'activities_limit' => array(
			'type' => 'limits_select',
			'namespace' => 'custom_activities',
			'min' => 0,
			'max' => 1000,
			'default' => 100
		),
		'tests_filter' => array(
			'namespace' => 'custom_tests'
		),
		'tests_columns' => array(
			'type' => 'columns_select',
			'namespace' => 'custom_tests',
			'default' => array(
				'tests:id' => 75,
				'cases:title' => 0
			)
		),
		'tests_limit' => array(
			'type' => 'limits_select',
			'namespace' => 'custom_tests',
			'min' => 0,
			'max' => 1000,
			'default' => 100
		),
		'content_hide_links' => array(
			'namespace' => 'custom_content',
		)
	);

	// The resources to copy to the output directory when generating a
	// report.
	private static $_resources = array(
		'images/report-assets/plan16.svg',
		'images/report-assets/run16.svg',
		'images/report-assets/help.svg',
		'images/report-assets/goal.svg',
		'images/report-assets/stats.svg',
		'images/report-assets/time.svg',
		'js/jquery.js',
		'js/fusioncharts.js',
		'js/fusioncharts.charts.js',
		'js/fusioncharts.theme.fusion.js',
		'js/chart.js',
		'styles/print.css',
		'styles/reset.css',
		'styles/view.css',
		'styles/font.css',
		'font/Barlow-Regular.ttf',
		'font/Barlow-Italic.ttf',
		'font/Barlow-Medium.ttf',
		'font/Barlow-MediumItalic.ttf',
		'font/Barlow-SemiBold.ttf',
		'font/Barlow-SemiBoldItalic.ttf',
		'font/Barlow-Bold.ttf',
		'font/Barlow-BoldItalic.ttf'
	);

	public function __construct()
	{
		parent::__construct();
		$this->_model = new Milestones_summary_with_reliability_growth_curve_model();
		$this->_model->init();
		$this->_controls = $this->create_controls(
			self::$_control_schema
		);
	}

	public function prepare_form($context, $validation)
	{
		// Assign the validation rules for the controls used on the
		// form.
		$this->prepare_controls($this->_controls, $context, 
			$validation);

		// Assign the validation rules for the fields on the form
		// that are not covered by the controls and are specific to
		// this report.
		$validation->add_rules(
			array(
				'custom_status_include' => array(
					'type' => 'bool',
					'default' => false
				),
				'custom_activities_include' => array(
					'type' => 'bool',
					'default' => false
				),
				'custom_progress_include' => array(
					'type' => 'bool',
					'default' => false
				),
				'custom_tests_include' => array(
					'type' => 'bool',
					'default' => false
				),
				'custom_reliability_growth_curve_include' => array(
					'type' => 'bool',
					'default' => false
				)
			)
		);

		if (request::is_post())
		{
			return;
		}

		// We assign the default values for the form depending on the
		// event. For 'add', we use the default values of this plugin.
		// For 'edit/rerun', we use the previously saved values of
		// the report/report job to initialize the form. Please note
		// that we prefix all fields in the form with 'custom_' and
		// that the storage format omits this prefix (validate_form).

		if ($context['event'] == 'add')
		{
			$defaults = array(
				'status_include' => true,
				'activities_include' => true,
				'progress_include' => true,
				'tests_include' => true,
				'reliability_growth_curve_include' => true
			);
		}
		else
		{
			$defaults = $context['custom_options'];
		}

		foreach ($defaults as $field => $value)
		{
			$validation->set_default('custom_' . $field, $value);
		}
	}

	public function validate_form($context, $input, $validation)
	{
		// We begin with validating the controls used on the form.
		$values = $this->validate_controls(
			$this->_controls,
			$context,
			$input,
			$validation);

		if (!$values)
		{
			return false;
		}

		static $fields = array(
			'status_include',
			'activities_include',
			'progress_include',
			'tests_include',
			'reliability_growth_curve_include'
		);

		// And then add our fields from the form input that are not
		// covered by the controls and return the data as it should be
		// stored in the report options.
		foreach ($fields as $field)
		{
			$key = 'custom_' . $field;
			$values[$field] = arr::get($input, $key);
		}

		return $values;
	}

	public function render_form($context)
	{
		$project = $context['project'];

		$params = array(
			'controls' => $this->_controls,
			'project' => $project,
			'test_columns' => $context['test_columns']
		);

		// Note that we return separate HTML snippets for the form/
		// options and the used dialogs (which must be included after
		// the actual form as they include their own <form> tags).
		return array(
			'form' => $this->render_view(
				'form',
				$params,
				true
			),
			'after_form' => $this->render_view(
				'form_dialogs',
				$params,
				true
			)
		);
	}

	public function run($context, $options)
	{
		$project = $context['project'];

		// We start by getting the milestone and the associated test
		// runs/plans from the database which also includes the full
		// statistics (status counts).
		$milestone = $this->_helper->get_milestone(
			$options['milestones_id']
		);

		if ($milestone)
		{
			$runs = $this->_helper->get_runs_by_milestone(
				$milestone->id
			);
		}
		else 
		{
			$runs = array();
		}

		$runs_noplan = array();
		foreach ($runs as $run)
		{
			if ($run->is_plan)
			{
				foreach ($run->runs as $r)
				{
					$runs_noplan[] = $r;					
				}
			}
			else
			{
				$runs_noplan[] = $run;
			}
		}

		// We then read the activity (data for the activity chart) and
		// the activities (test results and comments added over time).
		$activities_include = $options['activities_include'];
		$activities_limit = $options['activities_limit'];

		if ($activities_include && $milestone)
		{
			$this->_helper->get_daterange_tofrom(
				$options['activities_daterange'],
				$options['activities_daterange_from'],
				$options['activities_daterange_to'],
				$activities_from,
				$activities_to
			);

			$activity = $this->_helper->get_activity_by_milestone(
				$milestone,				
				$activities_from,
				$activities_to
			);

			// Get the statuses for the status filter, if any. If we
			// include all statuses, we can ignore this.
			if ($options['activities_statuses_include'] == 
				TP_REPORT_PLUGINS_STATUSES_ALL)
			{
				$activities_status_ids = null;	
			}
			else
			{
				$activities_status_ids = obj::get_ids(
					$this->_helper->get_statuses(
						$options['activities_statuses_ids']
					)
				);
			}

			$activities = $this->_helper->get_activities_by_milestone(
				$milestone,
				$activities_from,
				$activities_to,
				$activities_status_ids,
				$activities_limit,
				$activities_rels
			);
		}
		else
		{
			$activity = null;
			$activities = array();
			$activities_rels = array();
			$activities_from = null;
			$activities_to = null;
			$activities_status_ids = null;
		}

		// We then read the progress and forecast information (for the
		// burndown charts and forecast details).
		$progress_include = $options['progress_include'];

		$progress = null;
		$burndown = null;

		if ($progress_include && $milestone && $runs)
		{
			$progress = $this->_helper->get_progress_by_milestone(
				$milestone,
				$runs
			);

			if ($progress)
			{
				$burndown = $this->_helper->get_burndown_by_milestone(
					$milestone,
					$runs_noplan, // sic! Actual runs only
					$progress
				);
			}
		}

		$status_include = $options['status_include'];
		$reliability_growth_curve_include = $options['reliability_growth_curve_include'];

		$tests_result = $this->_model->get_test_results_for_reliability_growth_curve(
			array_column(
				$runs_noplan,
				'id'
			)
		);

		// Render the report to a temporary file and return the path
		// to TestRail (including additional resources that need to be
		// copied). We use a different view if the milestone no longer
		// exists.
		if ($milestone)
		{
			$view_name = 'index';
		}
		else 
		{
			$view_name = 'index_na';
		}

		return array(
			'resources' => self::$_resources,
			'html_file' => $this->render_page(
				$view_name,
				array(
					'report' => $context['report'],
					'project' => $project,
					'milestone' => $milestone,
					'runs' => $runs,
					'runs_noplan' => $runs_noplan,
					'status_include' => $status_include,
					'activities_include' => $activities_include,
					'activity' => $activity,
					'activities' => $activities,
					'activities_rels' => $activities_rels,
					'activities_from' => $activities_from,
					'activities_to' => $activities_to,
					'activities_limit' => $activities_limit,
					'progress_include' => $progress_include,
					'progress' => $progress,
					'burndown' => $burndown,
					'tests_include' => $options['tests_include'],
					'test_filters' => $options['tests_filters'],
					'test_limit' => $options['tests_limit'],
					'test_columns' => $context['test_columns'],
					'test_columns_for_user' => 
						$options['tests_columns'],
					'fields' => $context['fields'],
					'case_fields' => $context['case_fields'],
					'test_fields' => $context['test_fields'],
					'show_links' => !$options['content_hide_links'],
					'reliability_growth_curve_include' => $reliability_growth_curve_include,
					'tests_result' => $tests_result
				)
			)
		);
	}
}

class Milestones_summary_with_reliability_growth_curve_model extends BaseModel
{
	public function get_test_results_for_reliability_growth_curve($run_ids)
	{
		if (!$run_ids) {
			return array();
		}

		# list status (is_final)
		$query = $this->db->query(
			'
			SELECT
				statuses.id,
				statuses.name,
				statuses.is_final
			FROM
				statuses
			'
		);
		$statuses = $query->result();
		if (!$statuses) {
			return array();
		}

		$final_statuses = array();
		$final_statuses_name = array();
		foreach ($statuses as $status){
    		if ($status->is_final){
				$final_statuses[] = $status->id;
				$final_statuses_name[] = $status->name;
			}
		}

		if ('mysql' == DB_DRIVER) {
			$query = $this->db->query(
				'
				SELECT
					test_changes.status_id,
					test_changes.created_on,
					test_changes.test_id,
					test_changes.elapsed,
					test_changes.defects
				FROM
					test_changes
				FORCE INDEX(ix_test_changes_run_order)
				WHERE
					test_changes.run_id IN ({0}) AND test_changes.status_id IN ({1})
				ORDER BY
					test_changes.created_on DESC',
				$run_ids,
				$final_statuses
			);
		} else {
			$query = $this->db->query(
				'
				SELECT
					test_changes.status_id,
					test_changes.created_on,
					test_changes.test_id,
					test_changes.elapsed,
					test_changes.defects
				FROM
					test_changes
				WITH
					(INDEX(ix_test_changes_run_order))
				WHERE
					test_changes.run_id IN ({0}) AND test_changes.status_id IN ({1})
				ORDER BY
					test_changes.created_on DESC',
				$run_ids,
				$final_statuses
			);
		}

		$test_results_for_reliability_growth_curve = $query->result();
		if (!$test_results_for_reliability_growth_curve) {
			return array();
		}

		# 信頼度成長曲線用のデータ集計
		$first_day = null;
		$last_day = null;
		$total_test = 0;
		$total_elapsed = 0;
		$total_num_defect = 0;
		$total_elapsed_lst = array();
		$total_num_defect_lst = array();
		$total_test_lst = array();
		$elapsed_day_lst = array();
		$total_num_defect_daily_lst = array();
		$elapsed_day_before = -1;
		$total_elasped_day = 0;
		$timezone = new DateTimeZone('Asia/Tokyo');
		#$timezone = new DateTimeZone(date_default_timezone_get() ?: 'Asia/Tokyo');

		foreach ($test_results_for_reliability_growth_curve as $result) {
			// 経過日数（src）
			if ($last_day == null) {
				$last_day = new DateTime("@$result->created_on");
				$last_day->setTimezone($timezone);
			}

			$first_day = new DateTime("@$result->created_on");
			$first_day->setTimezone($timezone);

			// 経過日数（dst）
			$created_on = new DateTime("@$result->created_on");
			$created_on->setTimezone($timezone);
			$elapsed_day = $created_on->diff($last_day)->days;

			// 実施テスト数の累計
			$total_test++;

			// 経過時間の累計
			$curr_elapsed = $result->elapsed;
			if ($curr_elapsed !== null) {
				$total_elapsed += (int)$curr_elapsed;
			}

			// 欠陥数の累計
			if ($result->defects !== null) {
				$defects_array = explode(',', $result->defects);
				$total_num_defect += count($defects_array);
			}

			// 表示
			//$msg = sprintf('%s,%s,%s,"%s",%s,%s,%s,%s,%s', $result->test_id, $result->status_id, $curr_elapsed, $result->defects, date('Y-m-d H:i:s', $result->created_on), $total_test, $total_elapsed, $elapsed_day, $total_num_defect);
			//logger::debugr('[REPORT][DEBUG][report.php] get_test_results_for_reliability_growth_curve $msg', $msg);

			// グラフ(経過時間(m) x 欠陥数) データ
			$total_elapsed_lst[] = $total_elapsed;
			$total_num_defect_lst[] = $total_num_defect;

			// グラフ(テスト実施数 x 欠陥数) データ
			$total_test_lst[] = $total_test;

			// グラフ(経過日数 x 欠陥数) データ
			if ($elapsed_day > $elapsed_day_before) {
				$elapsed_day_lst[] = $elapsed_day;
				$total_num_defect_daily_lst[] = $total_num_defect;
				$elapsed_day_before = $elapsed_day;
				$total_elasped_day = $elapsed_day;
			}
		}

		// グラフ1のデータを準備：経過時間(m)
		$graph1_data = array();
		foreach ($total_elapsed_lst as $index => $elapsed) {
			$defects = $total_num_defect_lst[$index];
			$graph1_data[] = array(
				'label' => $elapsed / 60, # 分
				'value' => $defects
			);
		}

		// テスト実行時間は不連続のデータであるため補完する
		$graph1_data_imputed = [];
		$graph1_data_imputed[] = $graph1_data[0];
		for ($i = 1; $i < count($graph1_data); $i++) {
			$currentLabel = intval($graph1_data[$i]['label']);
			$previousLabel = intval($graph1_data[$i - 1]['label']);
			$difference = $currentLabel - $previousLabel;
			if ($difference > 1) {
				for ($j = 1; $j < $difference; $j++) {
					$interpolatedLabel = $previousLabel + $j;
					$graph1_data_imputed[] = [
						'label' => strval($interpolatedLabel),
						'value' => null
					];
				}
			}
			$graph1_data_imputed[] = $graph1_data[$i];
		}

		// グラフ2のデータを準備：経過日数
		$graph2_data = array();
		foreach ($elapsed_day_lst as $index => $day) {
			$defects = $total_num_defect_daily_lst[$index];
			$graph2_data[] = array(
				'label' => $day,
				'value' => $defects
			);
		}
		// テスト経過日数は不連続のデータであるため補完する
		$graph2_data_imputed = [];
		$graph2_data_imputed[] = $graph2_data[0];
		for ($i = 1; $i < count($graph2_data); $i++) {
			$currentLabel = intval($graph2_data[$i]['label']);
			$previousLabel = intval($graph2_data[$i - 1]['label']);
			$difference = $currentLabel - $previousLabel;
			if ($difference > 1) {
				for ($j = 1; $j < $difference; $j++) {
					$interpolatedLabel = $previousLabel + $j;
					$graph2_data_imputed[] = [
						'label' => strval($interpolatedLabel),
						'value' => null
					];
				}
			}
			$graph2_data_imputed[] = $graph2_data[$i];
		}

		// グラフ3のデータを準備：テスト実施数
		$graph3_data = array();
		foreach ($total_test_lst as $index => $test) {
			$defects = $total_num_defect_lst[$index];
			$graph3_data[] = array(
				'label' => $test,
				'value' => $defects
			);
		}

		$graph_data = array();
		$graph_data['first_day'] = $first_day;
		$graph_data['last_day'] = $last_day;
		$graph_data['total_num_test'] = $total_test;
		$graph_data['total_num_defect'] = $total_num_defect;
		$graph_data['total_elasped_day'] = $total_elasped_day;
		$graph_data['graph1_data'] = $graph1_data_imputed;
		$graph_data['graph2_data'] = $graph2_data_imputed;
		$graph_data['graph3_data'] = $graph3_data;
		$graph_data['final_statuses_name'] = $final_statuses_name;
		return $graph_data;
	}
}