"""
テストランに登録されたテスト結果を取得します。

使い方：
1. TestRail の [管理] > [サイト設定] > [API] タブで 'API の有効化' チェックボックスを有効化してください。
2. ログイン情報をご利用環境に合わせて変更してください。
3. 実行時引数を2つ指定してください。
   ・第一引数：テストランID
   ・第二引数：ファイルダウンロード先フォルダーパス (事前に用意してください。)
    実行例： python.exe get_result_run.py 5 C:\\sample
    ↑テストランID が 5 のテスト結果を C:\\sample にダウンロードします。
4. 指定したフォルダーパスに 'テストID\結果ID_ステータス' のフォルダーを作成して結果をダウンロードします。

出力されるファイル：
1. テスト結果(result.json)
2. テスト結果に添付されたファイル
3. コメント欄等に埋め込まれた画像ファイル
   TestRailが自動で採番したIDとファイル名を'_'で連結しファイル名としてダウンロードします。
   どこに埋め込まれた画像であるかは、「1. テスト結果」を確認してください。

その他：
1. 取得される添付ファイルのファイル名は以下のルールに従って出力されます。
   TestRailが自動で採番したIDとファイル名を'_'で連結しファイル名としてダウンロードします。
   どこに埋め込まれた画像であるかは、出力された「1. テスト結果」の内容を確認してください。
1. 全てのテスト結果を取得します。
   'Passed' や 'Failed' などのステータスや、日付で絞込みを行う場合 'get_results_for_run' のリクエストフィルターを使用してください。
   'get_results_for_run' の使用方法は以下のマニュアルを参照してください。
   https://docs.testrail.techmatrix.jp/api/reference/results/#get_results_for_run

"""

import sys
import os
import re
import json
import codecs
from testrail import *
from testrail_webclient import *

# 実行時引数の配列
# 第一引数：テストランID
# 第二引数：フォルダーパス
args = sys.argv

# TestRail ログイン情報 (★ ご利用環境に合わせて変更してください。)
testrail_url = "http://localhost:8000/"
user         = "user@techmatrix.co.jp"
password     = "password"

client = APIClient(testrail_url)
client.user = user
client.password = password

# テストランに含まれるテスト結果情報を取得 (★ 'get_results_for_run' のリクエストフィルター)
response_result_run = client.send_get('get_results_for_run/' + args[1] )

if 'offset' in response_result_run:
    dict_result_run = response_result_run['results']
    mode = "new"
else:
    dict_result_run = response_result_run
    mode = "old"

print(str(len(dict_result_run)) + " 件のテスト結果があります。")

# ステータスが 'Passed' のテスト結果から添付ファイルを取得する
# dict_result_run = client.send_get('get_results_for_run/' + args[1] + '&status_id=1')

# ステータス情報の取得
status_information = client.send_get('get_statuses')

# ステータス情報の ID と ラベルでステータスの辞書を作成
status_dict = {}
for one_status in status_information:
    status_dict[one_status.get('id')] = one_status.get('label')

# テスト結果の数分繰り返す
for i, one_result in enumerate(dict_result_run):
    # 結果ID
    result_id = one_result.get('id')

    # テストID
    test_id = one_result.get('test_id')

    print(str((i + 1)) + " 件目(" + str(len(dict_result_run)) + " 件中)を処理しています。ケースID " + str(test_id))

    # ステータスID
    result_status_id = one_result.get('status_id')

    # ステータスラベルを取得。ステータスIDが存在しない場合(Comment)は取得しない。
    if result_status_id in status_dict:
        result_status_label = status_dict[result_status_id]
    else:
        result_status_label = 'Comment'

    # ダウンロード先フォルダー
    download_folder = args[2] + '\\' + str(test_id) + '\\' + str(result_id) + '_' + result_status_label

    # フォルダーがある場合は作成しない
    if not os.path.isdir(download_folder):
        os.makedirs(download_folder)

    # テスト結果をJSON ⇒ 辞書型 ⇒ 文字列型に変換
    one_result_dict = json.dumps(one_result)
    one_result_str = json.dumps(one_result_dict)

    # テスト結果をファイル出力
    result_file_path = download_folder + '\\result.json'
    with codecs.open(result_file_path, 'w', "utf-8") as f:
        json.dump(one_result, f, ensure_ascii=False, indent=4)

    # 埋め込み画像情報を取得
    pattern = r'index.php\?/attachments/get/([\w-]+)'
    embedded_image_list = re.findall(pattern, one_result_str)

    # 埋め込み画像のID取得用
    embedded_image_list_id = []

    # 埋め込む画像数分繰り返す
    for one_embedded_image in embedded_image_list:
        # 埋め込み画像の ID
        embedded_image_id = re.sub(r'index.php\?/attachments/get/|\)', '', one_embedded_image)

        # 埋め込み画像の ID をリストに追加
        embedded_image_list_id.append(str(embedded_image_id))

        # ファイルをダウンロード
        webclient = WebClient(testrail_url, user, password)
        filename, content = webclient.get_attachement(str(embedded_image_id))

        # 添付ファイル保存
        with open(download_folder + '/' + embedded_image_id + '_' + filename, mode='wb') as f:
            f.write(content)

    # テストの添付ファイルのリストを取得 (リストには複数回のテスト結果に添付されたファイル情報が含まれる)
    list_attachment = client.send_get('get_attachments_for_test/' + str(test_id))

    # 添付ファイル数分繰り返す
    for one_attachment in list_attachment:

        # 結果IDが同じ場合だけ取得する
        do_download = False
        if mode == "new":
            if (str(result_id) == str(one_attachment.get('entity_id'))):
                do_download = True
        elif mode == "old":
            if (str(result_id) == str(one_attachment.get('result_id'))):
                do_download = True


        # ファイル取得       
        if do_download:
            # ファイルID
            file_id = str(one_attachment.get('id'))

            # ファイル ID が埋め込み画像の ID と一致する場合は取得しない
            if not file_id in embedded_image_list_id:
                # ファイルをダウンロード
                webclient = WebClient(testrail_url, user, password)
                filename, content = webclient.get_attachement(file_id)

                # 添付ファイル保存
                with open(download_folder + '/' + file_id + '_' + filename, mode='wb') as f:
                    f.write(content)
