import sys
import json
import csv
from copy import deepcopy
from datetime import datetime, timedelta
from testrail import *

# TestRail ログイン情報 ※ご利用環境に合わせて変更してください。
testrail_url = 'http://localhost/'
user         = 'testrail-user@domain.com'
password     = 'password'

# TestRail固有情報
system_field_key_label = [
    {'key': 'id', 'label': 'ケースID', 'type_id': 2, 'id': 0},
    {'key': 'title', 'label': 'タイトル', 'type_id': 1, 'id': 0},
    {'key': 'section_id', 'label': 'セクション', 'type_id': 1, 'id': 0},
    {'key': 'template_id', 'label': 'テンプレートID', 'type_id': 2, 'id': 0},
    {'key': 'type_id', 'label': 'タイプ', 'type_id': 0, 'id': 0}, 
    {'key': 'priority_id', 'label': '優先度', 'type_id': 0, 'id': 0},
    {'key': 'milestone_id', 'label': 'マイルストーンID', 'type_id': 2, 'id': 0},
    {'key': 'refs', 'label': '参照', 'type_id': 1, 'id': 0},
    {'key': 'created_on', 'label': '作成日', 'type_id': 8, 'id': 0},
    {'key': 'created_by', 'label': '作成者', 'type_id': 7, 'id': 0},
    {'key': 'updated_on', 'label': '更新日', 'type_id': 8, 'id': 0},
    {'key': 'updated_by', 'label': '更新者', 'type_id': 7, 'id': 0},
    {'key': 'estimate', 'label': '見積もり', 'type_id': 1, 'id': 0},
    {'key': 'estimate_forecast', 'label': '予想見積もり', 'type_id': 1, 'id': 0}, 
    {'key': 'suite_id', 'label': 'テストスイート', 'type_id': 1, 'id': 0},
    {'key': 'is_deleted', 'label': 'is_deleted', 'type_id': 1, 'id': 0},
    {'key': 'display_order', 'label': 'display_order', 'type_id': 1, 'id': 0},
    {'key': 'custom_steps_separated(content)', 'label': '手順(Step)', 'type_id': 3, 'id': 0},
    {'key': 'custom_steps_separated(expected)', 'label': '期待する結果(Step)', 'type_id': 3, 'id': 0},
    #{'key': 'custom_steps_separated(additional_info)', 'label': '補足情報(Step)', 'type_id': 3, 'id': 0},
    #{'key': 'custom_steps_separated(refs)', 'label': '参照(Step)', 'type_id': 3, 'id': 0},
    {'key': 'content', 'label': '手順', 'type_id': 3, 'id': 0},
    {'key': 'expected', 'label': '期待する結果', 'type_id': 3, 'id': 0},
]

default_fieldnames = [
    'id', 'title', 'custom_steps_separated(content)', 'custom_steps_separated(expected)', 
    #'custom_steps_separated(additional_info)', 'custom_steps_separated(refs)'
    #'custom_steps_separated', 'custom_steps_separated(content)', 'custom_steps_separated(expected)', 'custom_steps_separated(additional_info)', 'custom_steps_separated(refs)'
]

def log_debug(message):
    current_time = datetime.now().strftime("[%Y-%m-%d %H:%M:%S]")
    log_message = f"{current_time}[INFO] {message}"
    print(log_message)

class TestRailTestcaseExporter:
    def __init__(self, base_url, user, password, charcode):
        self.client = APIClient(base_url)
        self.client.user = user
        self.client.password = password
        self.charcode = charcode
        self.case_field = None
        self.result_field = None
        self.template = None
        self.priority = None
        self.case_type = None
        self.milestone = None
        self.status = None
        self.users = []
        self.section = None
        self.is_admin = None


    def run(self, testsuite_id, output_csv_file):
        self.testsuite_id = testsuite_id
        log_debug("Read TestRail field data.")
        self._config()
        log_debug("Read TestRail testcase data.")
        testcase_data = self._load()
        log_debug("Write data to CSV file.")
        self._write(testcase_data, output_csv_file)


    # TestRailのカスタムフィールド、タイプの情報を取得
    def _config(self):
        # プロジェクトID, テストラン名
        testsuite_information = self._send_get('get_suite/' + self.testsuite_id, 'results')
        self.project_id = str(testsuite_information['project_id'])
        self.testsuite_name = testsuite_information['name']
        log_debug(f"  -> get_run: project_id={self.project_id}, testsuite_name=\"{self.testsuite_name}\"")

        # ケースフィールド情報
        log_debug(f"  -> get_case_fields")
        self.case_field = self._send_get('get_case_fields', 'results')

        """
        print("=== get_case_fields ===")
        print(json.dumps(self.case_field, ensure_ascii=False, indent=4))
        """

        self.key_label = system_field_key_label

        for case_field in self.case_field:
            if case_field['is_active'] == True:
                temp = {}
                temp['id'] = case_field['id']
                temp['key'] = case_field['system_name']
                temp['label'] = case_field['label']
                temp['type_id'] = case_field['type_id']
                self.key_label.append(temp)

        log_debug(f"  -> done.")


    def _load(self):
        cases = self._send_get('get_cases/' + self.project_id + '&suite_id=' + self.testsuite_id, 'cases')
        log_debug(f"  -> get_cases: project_id={self.project_id}, suite_id={self.testsuite_id} nums={len(cases)}")

        return cases


    def _write(self, data, output_csv_file):
        fixed_keys_separated = ['content', 'expected']#, 'additional_info', 'refs']

        # マッピング用の辞書を作成
        key_to_label = {mapping['key']: mapping['label'] for mapping in self.key_label}

        log_debug(f"  -> open csv file ({output_csv_file})")

        with open(output_csv_file, 'w', newline='', encoding=self.charcode) as csvfile:

            # 全てのキーを抽出
            all_keys = deepcopy(default_fieldnames)
            for record in list(data[0]):
                if record not in all_keys:
                    all_keys.append(record)

            # フィールド名をラベルに変換
            label_fieldnames = [key_to_label.get(key, key) for key in all_keys]

            writer = csv.DictWriter(csvfile, fieldnames=label_fieldnames, extrasaction='ignore')
            writer.writeheader()

            data.sort(key=lambda x: x['id']) # 'id'で昇順ソート

            progress_threshold = 1 if len(data) < 10 else round(len(data)/10)
            log_debug(f"  -> write csv data... ({len(data)} elems)")
            for count, record in enumerate(data):
                if count != 0 and count%progress_threshold == 0:
                    log_debug(f"  -> {round(count/progress_threshold)*10} % ({count} elems)")

                steps = record.get('custom_steps_separated', [])
                new_record = {}
                for k in ['id', 'title']:
                    for item in self.key_label:
                        if item['key'] == k:
                            type_id = item['type_id']
                            field_id = item['id']
                    new_record[key_to_label.get(k, k)] = self._get_actual_value(k, type_id, field_id, record[k])

                if steps:
                    new_record[key_to_label.get('custom_steps_separated')] = self._format_steps(steps, fixed_keys_separated)
                    first_step = steps[0]
                    for k in fixed_keys_separated:
                        actual_key = f'custom_steps_separated({k})'
                        new_record[key_to_label.get(actual_key)] = self._get_actual_value(k, 0, 0, first_step.get(k, '<none>'))
                    
                # fieldnamesにないその他のフィールドを取得
                extra_keys = [k for k in list(record.keys()) if k not in default_fieldnames]
                for extra_key in extra_keys:
                    if extra_key == "custom_steps_separated":
                        continue
                    for item in self.key_label:
                        if item['key'] == extra_key:
                            type_id = item['type_id']
                            field_id = item['id']
                    new_record[key_to_label.get(extra_key)] = self._get_actual_value(extra_key, type_id, field_id, record[extra_key])
                
                writer.writerow(new_record)

                # 出力行の生成
                if steps:
                    for step in steps[1:]:
                        new_row = {}
                        for k in fixed_keys_separated:
                            actual_key = f'custom_steps_separated({k})'
                            new_row[key_to_label.get(actual_key)] = self._get_actual_value(k, 0, 0, step.get(k, '<none>'))

                        writer.writerow(new_row)
        log_debug(f"  -> 100 % ({len(data)} elems)")
        log_debug(f"  -> done.")


    def _send_get(self, api, key):
        response = self.client.send_get(api)
        if 'offset' not in response:
            return response
        else:
            return response[key]


    def _format_steps(self, steps, fixed_keys):
        lines = []
        for idx, step in enumerate(steps):
            line = f'\n<{idx + 1}>'
            for key in fixed_keys:
                value = step.get(key, '<none>')
                label = key
                if key == 'status_id':
                    value = self._get_actual_value(key, 0, 0, value)
                for item in self.key_label:
                    if item['key'] == key:
                        label = item['label']
                line += f'\n[{label}]:\n{value}'
            lines.append(line)
        return '\n'.join(lines)
    

    """
    # 添付ファイル取得
    def _get_attachement(self, attachment_id):
        # セッションを作成
        self.__session = requests.Session()

        # ログインのPOSTデータ
        login_data = {
            "name": self.user,
            "password": self.password,
        }

        # ログイン実行
        login_response = self.__session.post(self.__auth_url, data=login_data)
        if login_response.status_code > 201:
            try:
                error = login_response.json()
            except:
                error = str(login_response.content)
            raise WebError('TestRail Web(login) returned HTTP %s (%s)' % (login_response.status_code, error))

        # 添付ファイル取得
        attachment_url = self.__base_url + 'index.php?/attachments/get/' + str(attachment_id)
        response = self.__session.get(attachment_url)
        if response.status_code > 201:
            try:
                error = response.json()
            except:
                error = str(response.content)
            raise WebError('TestRail Web returned HTTP %s (%s)' % (response.status_code, error))

        content_disposition = response.headers.get('Content-Disposition', '')

        # ファイル名のURLエンコード部分を取得
        filename_match = re.search(r"filename\*=(UTF-8'')?(.+)", content_disposition)

        # ファイル名
        filename = ""
        if filename_match:
            encoded_filename = filename_match.group(2)
            filename = unquote(encoded_filename)

        return filename,response.content
    """


    def _find_name_by_id(self, data, key, target_id):
        for record in data:
            if record['id'] == target_id:
                return record[key]
        log_debug(f"_find_name_by_id: There is no matched target id. {key}, {target_id}")
        return None


    def _find_value_by_id_and_index(self, data, field_id, num):
        # 指定されたIDに一致する辞書を探す
        target_dict = next((item for item in data if item['id'] == field_id), None)
        
        if target_dict is None:
            return None
        
        # configsが存在し、その中にoptionsが存在し、その中にitemsが存在するか確認
        configs = target_dict.get('configs')
        if not configs:
            return None
        
        options = configs[0].get('options')
        if not options:
            return None
        
        items_str = options.get('items')
        if not items_str:
            return None

        # itemsを解析して、指定されたnumに対応する値を探す
        items_list = items_str.split('\n')
        for item in items_list:
            num_str, value = [element.strip() for element in item.split(',')]
            if int(num_str) == num:
                return value
        
        return None


    def _find_value(self, field_id, index):
        value = self._find_value_by_id_and_index(self.case_field, field_id, index)
        if not value:
            value = self._find_value_by_id_and_index(self.result_field, field_id, index)
        return value


    def _is_admin_user(self):
        if self.is_admin == None:
            self_user_information = self._send_get('get_user_by_email&email='+self.client.user, 'results')
            self.is_admin = self_user_information['is_admin']
            if self._is_admin_user() == False:
                log_debug("Current user is not Admin user. Username is not converted to the user ID.")
        return self.is_admin

    def _get_username(self, userid):
        if self._is_admin_user() == False:
            return userid

        if len(self.users) > 0:
            for user in self.users:
                if userid == user['id']:
                    return user['name']

        user_information = self._send_get('get_user/'+str(userid), 'results')
        user = {}
        user['id'] = userid
        user['name'] = user_information['name']
        self.users.append(user)

        return user['name']
    

    def _get_actual_value(self, system_name, type_id, field_id, raw_value):

        if not raw_value:
            return None
        
        # 特殊なケース1: テンプレート
        if system_name == 'template_id':
            if not self.template:
                self.template = self._send_get(f'get_templates/{self.project_id}', 'results')
            return self._find_name_by_id(self.template, 'name', raw_value)

        # 特殊なケース2: 添付ファイルID ※添付ファイル名やファイル保存は未対応
        elif system_name == 'attachment_ids':
            return f"\"{', '.join(map(str, raw_value))}\")"

        # 特殊なケース3: 優先度
        elif system_name == 'priority_id':
            if not self.priority:
                self.priority = self._send_get('get_priorities', 'results')
            return self._find_name_by_id(self.priority, 'name', raw_value)

        # 特殊なケース4: タイプ
        elif system_name == 'type_id':
            if not self.case_type:
                self.case_type = self._send_get('get_case_types', 'results')
            return self._find_name_by_id(self.case_type, 'name', raw_value)

        # 特殊なケース5: ステータス
        elif system_name == 'status_id':
            if not self.status:
                self.status = self._send_get('get_statuses', 'results')
            status = self._find_name_by_id(self.status, 'label', int(raw_value))
            return status

        # 特殊なケース6: セクション
        elif system_name == 'section_id':
            if not self.section:
                self.section = self._send_get('get_sections/' + self.project_id + '&suite_id=' + self.testsuite_id, 'sections')
            status = self._find_name_by_id(self.section, 'name', raw_value)
            return status

        # Dropdown
        elif type_id == 6:
            value = self._find_value(field_id, raw_value)
            if not value:
                value = raw_value
            return value

        # Multi-select
        elif type_id == 12:
            elems = []
            for elem in raw_value:
                elems.append(self._find_value(field_id, elem))
            if elems:
                return ', '.join(elems)
            else:
                return None
 
        # User
        elif type_id == 7:
            value = self._get_username(raw_value)
            if not value:
                value = raw_value
            return value

        # Date
        elif type_id == 8:
            if isinstance(raw_value, str):
                return raw_value
            else:
                dt_object = datetime.utcfromtimestamp(raw_value) + timedelta(hours=9)
                return dt_object.strftime('%Y/%m/%d %H:%M:%S')

        # Milestone
        elif type_id == 9:
            if not self.milestone:
                self.milestone = self._send_get(f'get_milestones/{self.project_id}', 'milestones')
            return self._find_name_by_id(self.milestone, 'name', raw_value)

        # BDD Scenario
        elif type_id == 13:
            value = ''
            raw_value_json = json.loads(raw_value)
            for elem in raw_value_json:
                value += elem['content'] + '\n'
            return value

        # BDD Scenario Result
        elif type_id == 14:
            value = ''
            raw_value_json = json.loads(raw_value)
            for elem in raw_value_json:
                value += f"内容:\n{elem['content']}\n\n実際の結果:\n{elem['actual']}\n\nステータス:\n{self._get_actual_value('status_id', 0, 0, elem['status_id'])}\n\n"
            return value

        # その他（テキスト）
        else:
            return raw_value

class WebError(Exception):
    pass

#---------------------------------------------------------------------------------
# main

instruction = """

 ### TestRail テストランCSVエクスポート ###

  テストスイートに含まれるテストケースをCSVファイルとして出力します。

  使い方：
    > python  tr_case_csv_export.py  TESTSUITE_ID  OUTPUT_CSV_FILENAME
    * TESTSUITE_ID: テストスイートID（対象のテストスイートをWebブラウザで開いた際に表示される index.php?/suites/view/xxx のxxxの数字）
    * OUTPUT_CSV_FILENAME: 出力するCSVファイル名

  実行例：
    > python  tr_case_csv_export.py  18  testcase_18.csv
    * テストスイートID が 18 のテスト結果を testcase_18.csv としてCSVファイルに出力します。

  事前準備：
    1. TestRail の [管理] > [サイト設定] > [API] タブで 'API の有効化' チェックボックスを有効化してください。
    2. スクリプト内の上部に記載されたTestRailのログイン情報をご利用環境に合わせて変更してください。

"""

def main():
    args = sys.argv
    if len(args) != 3:
        print(instruction)
        exit(0)

    testsuite_id = args[1]
    output_csv_file = args[2]
    charcode = 'cp932'
    client = TestRailTestcaseExporter(testrail_url, user, password, charcode)
    client.run(testsuite_id, output_csv_file)


if __name__ == "__main__":
    main()
