import sys
import json
import csv
from copy import deepcopy
from datetime import datetime, timedelta
from testrail import *

# TestRail ログイン情報 ※ご利用環境に合わせて変更してください。
testrail_url = 'http://localhost/'
user         = 'testrail-user@domain.com'
password     = 'password'

# TestRail固有情報
system_field_key_label = [
    {'key': 'id', 'label': 'ケースID', 'type_id': 2, 'id': 0},
    {'key': 'test_id', 'label': 'テストID', 'type_id': 2, 'id': 0},
    {'key': 'title', 'label': 'タイトル', 'type_id': 1, 'id': 0},
    {'key': 'status_id', 'label': 'ステータス', 'type_id': 0, 'id': 0},
    {'key': 'created_on', 'label': '作成日', 'type_id': 8, 'id': 0},
    {'key': 'assignedto_id', 'label': '担当者', 'type_id': 7, 'id': 0},
    {'key': 'comment', 'label': 'コメント', 'type_id': 3, 'id': 0},
    {'key': 'version', 'label': 'バージョン', 'type_id': 1, 'id': 0},
    {'key': 'elapsed', 'label': '経過時間', 'type_id': 1, 'id': 0},
    {'key': 'defects', 'label': '欠陥', 'type_id': 1, 'id': 0},
    {'key': 'created_by', 'label': '作成者', 'type_id': 7, 'id': 0},
    {'key': 'attachment_ids', 'label': '添付ファイルID', 'type_id': 0, 'id': 0},
    {'key': 'case_id', 'label': 'ケース番号', 'type_id': 2, 'id': 0},
    {'key': 'run_id', 'label': 'テストランID', 'type_id': 2, 'id': 0},
    {'key': 'template_id', 'label': 'テンプレートID', 'type_id': 2, 'id': 0},
    {'key': 'type_id', 'label': 'タイプ', 'type_id': 0, 'id': 0}, 
    {'key': 'priority_id', 'label': '優先度', 'type_id': 0, 'id': 0},
    {'key': 'estimate', 'label': '見積もり', 'type_id': 1, 'id': 0},
    {'key': 'estimate_forecast', 'label': '予想見積もり', 'type_id': 1, 'id': 0}, 
    {'key': 'refs', 'label': '参照', 'type_id': 1, 'id': 0},
    {'key': 'milestone_id', 'label': 'マイルストーンID', 'type_id': 2, 'id': 0},
    {'key': 'custom_steps_separated', 'label': '手順', 'type_id': 3, 'id': 0},
    {'key': 'custom_step_results', 'label': '結果', 'type_id': 3, 'id': 0},
    {'key': 'custom_step_results(content)', 'label': '結果（手順）', 'type_id': 3, 'id': 0},
    {'key': 'custom_step_results(expected)', 'label': '結果（期待する結果）', 'type_id': 3, 'id': 0},
    {'key': 'custom_step_results(actual)', 'label': '結果（実際の結果）', 'type_id': 3, 'id': 0},
    {'key': 'custom_step_results(status_id)', 'label': '結果（ステータス）', 'type_id': 3, 'id': 0},
    {'key': 'content', 'label': '手順', 'type_id': 3, 'id': 0},
    {'key': 'expected', 'label': '期待する結果', 'type_id': 3, 'id': 0},
    {'key': 'actual', 'label': '実際の結果', 'type_id': 3, 'id': 0},
]

exclude_field_key = [
    'custom_steps_separated', 'sections_display_order', 'cases_display_order', 
]

default_fieldnames = [
    'id', 'test_id', 'title', 'status_id', 'custom_step_results',
    'custom_step_results(content)', 'custom_step_results(expected)',
    'custom_step_results(actual)', 'custom_step_results(status_id)'
]

def log_debug(message):
    current_time = datetime.now().strftime("[%Y-%m-%d %H:%M:%S]")
    log_message = f"{current_time}[INFO] {message}"
    print(log_message)

class TestRailTestRunExporter:
    def __init__(self, base_url, user, password, charcode):
        self.client = APIClient(base_url)
        self.client.user = user
        self.client.password = password
        self.charcode = charcode
        self.case_field = None
        self.result_field = None
        self.template = None
        self.priority = None
        self.case_type = None
        self.milestone = None
        self.status = None
        self.users = []
        self.is_admin = None


    def run(self, id, output_csv_file, mode):
        log_debug("Read TestRail field data.")
        self._config()

        testrun_ids = self.get_testruns_ids(id, mode)
        log_debug(f"  -> project_id={self.project_id}")

        if len(testrun_ids) == 0:
            log_debug(f" No test run.")
            exit(0)
        
        log_debug("Read TestRail TestRun data.")
        testrun_data = []
        for testrun_id in testrun_ids:
            data = self._load(testrun_id)
            testrun_data.extend(data)

        log_debug("Write data to CSV file.")
        self._write(testrun_data, output_csv_file)


    def get_testruns_ids(self, id, mode):
        if mode == 'milestone':
            log_debug(f"Mode: milestone")
            return self._get_runs_from_milestone(id)
        elif mode == 'testplan':
            log_debug(f"Mode: testplan")
            return self._get_runs_from_plan(id)
        else:
            log_debug(f"Mode: testrun")
            # プロジェクトIDを取得
            testrun_information = self._send_get('get_run/' + str(id), 'results')
            self.project_id = testrun_information['project_id']
            return [id]


    def _get_runs_from_milestone(self, milestone_id):
        # マイルストーンIDから子マイルストーンを含むリストを作成
        milestones_dict = self._send_get('get_milestone/' + str(milestone_id), "results")
        self.project_id = milestones_dict["project_id"]

        child_milestones_list = milestones_dict.get('milestones')
        milestones_list_id = [d.get('id') for d in child_milestones_list]
        milestones_list_id.append(int(milestone_id))

        run_ids = []
        for id in milestones_list_id:

            # マイルストーンに含まれるテストランを取得
            milestone_testruns_dict = self._send_get('get_runs/' + str(self.project_id) + '&milestone_id=' + str(id), "runs")
            if milestone_testruns_dict != None:
                milestone_testruns_id_list = [d.get('id') for d in milestone_testruns_dict]
                run_ids.extend(milestone_testruns_id_list)

            # マイルストーンに含まれるテスト計画を取得
            testplans_dict = self._send_get('get_plans/' + str(self.project_id) + '&milestone_id=' + str(id), "plans")
            plan_list_id = [d.get('id') for d in testplans_dict]
            for plan_id in plan_list_id:
                # テスト計画に含まれるテストランを取得
                testplan_entries_list = self._send_get('get_plan/' + str(plan_id), "")
                for run in testplan_entries_list['entries'][0]['runs']:
                    run_ids.append(run["id"])

        return run_ids


    def _get_runs_from_plan(self, plan_id):
        run_ids = []
        # テスト計画に含まれるテストランを取得
        testplan_entries_list = self._send_get('get_plan/' + str(plan_id), "")
        self.project_id = testplan_entries_list["project_id"]
        for run in testplan_entries_list['entries'][0]['runs']:
            run_ids.append(run["id"])

        return run_ids


    # TestRailのカスタムフィールド、タイプの情報を取得
    def _config(self):
        # ケースフィールド情報
        log_debug(f"  -> get_case_fields")
        self.case_field = self._send_get('get_case_fields', 'results')

        # 結果フィールド情報
        log_debug(f"  -> get_result_fields")
        self.result_field = self._send_get('get_result_fields', 'results')

        """
        print("=== get_case_fields ===")
        print(json.dumps(self.case_field, ensure_ascii=False, indent=4))
        print("=== get_result_fields ===")
        print(json.dumps(self.result_field, ensure_ascii=False, indent=4))
        """

        self.key_label = system_field_key_label

        for case_field in self.case_field:
            if case_field['is_active'] == True:
                temp = {}
                temp['id'] = case_field['id']
                temp['key'] = case_field['system_name']
                temp['label'] = case_field['label']
                temp['type_id'] = case_field['type_id']
                self.key_label.append(temp)

        for result_field in self.result_field:
            if result_field['is_active'] == True:
                temp = {}
                temp['id'] = result_field['id']
                temp['key'] = result_field['system_name']
                temp['label'] = result_field['label']
                temp['type_id'] = result_field['type_id']
                self.key_label.append(temp)

        log_debug(f"  -> done.")



    def _load(self, testrun_id):
        tests = self._send_get('get_tests/' + str(testrun_id), 'tests')
        log_debug(f"  -> get_tests: testrun_id={testrun_id}, nums={len(tests)}")

        run_results = self._send_get('get_results_for_run/' + str(testrun_id), 'results')
        log_debug(f"  -> get_results_for_run: testrun_id={testrun_id}, nums={len(run_results)}")

        self._merge_data(run_results, tests)
        log_debug(f"  -> merged case and result {len(run_results)} elems")
        log_debug(f"  -> done.")
        return run_results


    def _write(self, data, output_csv_file):
        fixed_keys_results = ['content', 'expected', 'actual', 'status_id']

        # マッピング用の辞書を作成
        key_to_label = {mapping['key']: mapping['label'] for mapping in self.key_label}

        log_debug(f"  -> open csv file ({output_csv_file})")

        with open(output_csv_file, 'w', newline='', encoding=self.charcode) as csvfile:

            # 全てのキーを抽出
            all_keys = deepcopy(default_fieldnames)
            for record in list(data[0]):
                if record not in all_keys:
                    all_keys.append(record)

            # フィールド名をラベルに変換
            label_fieldnames = [key_to_label.get(key, key) for key in all_keys]

            writer = csv.DictWriter(csvfile, fieldnames=label_fieldnames, extrasaction='ignore')
            writer.writeheader()

            data.sort(key=lambda x: x['id']) # 'id'で昇順ソート

            progress_threshold = 1 if len(data) < 10 else round(len(data)/10)
            log_debug(f"  -> write csv data... ({len(data)} elems)")
            for count, record in enumerate(data):
                if count != 0 and count%progress_threshold == 0:
                    log_debug(f"  -> {round(count/progress_threshold)*10} % ({count} elems)")
                
                steps = record.get('custom_step_results', [])
                new_record = {}
                for k in ['id', 'test_id', 'title', 'status_id']:
                    for item in self.key_label:
                        if item['key'] == k:
                            type_id = item['type_id']
                            field_id = item['id']
                    if k in record:
                        new_record[key_to_label.get(k, k)] = self._get_actual_value(k, type_id, field_id, record[k])

                if steps:
                    new_record[key_to_label.get('custom_step_results')] = self._format_steps(steps, fixed_keys_results)
                    first_step = steps[0]
                    for k in fixed_keys_results:
                        actual_key = f'custom_step_results({k})'
                        new_record[key_to_label.get(actual_key)] = self._get_actual_value(k, 0, 0, first_step.get(k, '<none>'))

                # fieldnamesにないその他のフィールドを取得
                extra_keys = [k for k in list(record.keys()) if k not in default_fieldnames]
                for extra_key in extra_keys:
                    for item in self.key_label:
                        if item['key'] == extra_key:
                            type_id = item['type_id']
                            field_id = item['id']
                    new_record[key_to_label.get(extra_key)] = self._get_actual_value(extra_key, type_id, field_id, record[extra_key])
                    
                writer.writerow(new_record)

                # 出力行の生成
                if steps:
                    for step in steps[1:]:
                        new_row = {}
                        for k in fixed_keys_results:
                            actual_key = f'custom_step_results({k})'
                            new_row[key_to_label.get(actual_key)] = self._get_actual_value(k, 0, 0, step.get(k, '<none>'))

                        writer.writerow(new_row)
        log_debug(f"  -> 100 % ({len(data)} elems)")
        log_debug(f"  -> done.")


    def _send_get(self, api, key):
        response = self.client.send_get(api)
        if 'offset' not in response:
            return response
        else:
            return response[key]


    def _merge_data(self, result, test):
        test_data_dict = {item['id']: item for item in test}
        for item in result:
            test_id = item.get('test_id')
            if test_id is not None and test_id in test_data_dict:
                for key, value in test_data_dict[test_id].items():
                    if key not in item and key not in exclude_field_key:
                        item[key] = value

        # テストには存在するが結果に存在しない場合：ラン作成後に追加されたケースにUntestedが付与されない不具合の対応
        # -> テストの情報のみを結果に追加する。
        test_result_dict = {item['test_id']: item for item in result}
        missing_untested_test_ids = set(test_data_dict) - set(test_result_dict)
        for missing_untested_test in test:
            if missing_untested_test["id"] in missing_untested_test_ids:
                # 結果のtest_idをテストのidに変更
                missing_untested_test["test_id"] = missing_untested_test["id"]
                # 結果のidを削除（None）
                missing_untested_test["id"] = 0
                result.append(missing_untested_test)


    def _format_steps(self, steps, fixed_keys):
        lines = []
        for idx, step in enumerate(steps):
            line = f'\n<{idx + 1}>'
            for key in fixed_keys:
                value = step.get(key, '<none>')
                label = key
                if key == 'status_id':
                    value = self._get_actual_value(key, 0, 0, value)
                for item in self.key_label:
                    if item['key'] == key:
                        label = item['label']
                line += f'\n[{label}]:\n{value}'
            lines.append(line)
        return '\n'.join(lines)
    

    """
    # 添付ファイル取得
    def _get_attachement(self, attachment_id):
        # セッションを作成
        self.__session = requests.Session()

        # ログインのPOSTデータ
        login_data = {
            "name": self.user,
            "password": self.password,
        }

        # ログイン実行
        login_response = self.__session.post(self.__auth_url, data=login_data)
        if login_response.status_code > 201:
            try:
                error = login_response.json()
            except:
                error = str(login_response.content)
            raise WebError('TestRail Web(login) returned HTTP %s (%s)' % (login_response.status_code, error))

        # 添付ファイル取得
        attachment_url = self.__base_url + 'index.php?/attachments/get/' + str(attachment_id)
        response = self.__session.get(attachment_url)
        if response.status_code > 201:
            try:
                error = response.json()
            except:
                error = str(response.content)
            raise WebError('TestRail Web returned HTTP %s (%s)' % (response.status_code, error))

        content_disposition = response.headers.get('Content-Disposition', '')

        # ファイル名のURLエンコード部分を取得
        filename_match = re.search(r"filename\*=(UTF-8'')?(.+)", content_disposition)

        # ファイル名
        filename = ""
        if filename_match:
            encoded_filename = filename_match.group(2)
            filename = unquote(encoded_filename)

        return filename,response.content
    """


    def _find_name_by_id(self, data, key, target_id):
        for record in data:
            if record['id'] == target_id:
                return record[key]
        log_debug(f"_find_name_by_id: There is no matched target id. {key}, {target_id}")
        return None


    def _find_value_by_id_and_index(self, data, field_id, num):
        # 指定されたIDに一致する辞書を探す
        target_dict = next((item for item in data if item['id'] == field_id), None)
        
        if target_dict is None:
            return None
        
        # configsが存在し、その中にoptionsが存在し、その中にitemsが存在するか確認
        configs = target_dict.get('configs')
        if not configs:
            return None
        
        options = configs[0].get('options')
        if not options:
            return None
        
        items_str = options.get('items')
        if not items_str:
            return None
        
        # itemsを解析して、指定されたnumに対応する値を探す
        items_list = items_str.split('\n')
        for item in items_list:
            num_str, value = [element.strip() for element in item.split(',')]
            if int(num_str) == num:
                return value
        
        return None


    def _find_value(self, field_id, index):
        value = self._find_value_by_id_and_index(self.case_field, field_id, index)
        if not value:
            value = self._find_value_by_id_and_index(self.result_field, field_id, index)
        return value


    def _is_admin_user(self):
        if self.is_admin == None:
            self_user_information = self._send_get('get_user_by_email&email='+self.client.user, 'results')
            self.is_admin = self_user_information['is_admin']
            if self._is_admin_user() == False:
                log_debug("Current user is not Admin user. Username is not converted to the user ID.")
        return self.is_admin

    def _get_username(self, userid):
        if self._is_admin_user() == False:
            return userid

        if len(self.users) > 0:
            for user in self.users:
                if userid == user['id']:
                    return user['name']

        user_information = self._send_get('get_user/'+str(userid), 'results')
        user = {}
        user['id'] = userid
        user['name'] = user_information['name']
        self.users.append(user)

        return user['name']


    def _get_actual_value(self, system_name, type_id, field_id, raw_value):

        if not raw_value:
            return None
        
        # 特殊なケース1: テンプレート
        if system_name == 'template_id':
            if not self.template:
                self.template = self._send_get(f'get_templates/{self.project_id}', 'results')
            return self._find_name_by_id(self.template, 'name', raw_value)

        # 特殊なケース2: 添付ファイルID ※添付ファイル名やファイル保存は未対応
        elif system_name == 'attachment_ids':
            return f"\"{', '.join(map(str, raw_value))}\")"

        # 特殊なケース3: 優先度
        elif system_name == 'priority_id':
            if not self.priority:
                self.priority = self._send_get('get_priorities', 'results')
            return self._find_name_by_id(self.priority, 'name', raw_value)

        # 特殊なケース4: タイプ
        elif system_name == 'type_id':
            if not self.case_type:
                self.case_type = self._send_get('get_case_types', 'results')
            return self._find_name_by_id(self.case_type, 'name', raw_value)

        # 特殊なケース5: ステータス
        elif system_name == 'status_id':
            if not self.status:
                self.status = self._send_get('get_statuses', 'results')
            status = self._find_name_by_id(self.status, 'label', int(raw_value))
            return status

        # Dropdown
        elif type_id == 6:
            value = self._find_value(field_id, raw_value)
            if not value:
                value = raw_value
            return value

        # Multi-select
        elif type_id == 12:
            elems = []
            for elem in raw_value:
                elems.append(self._find_value(field_id, elem))
            if elems:
                return ', '.join(elems)
            else:
                return None
 
        # User
        elif type_id == 7:
            value = self._get_username(raw_value)
            if not value:
                value = raw_value
            return value

        # Date
        elif type_id == 8:
            if isinstance(raw_value, str):
                return raw_value
            else:
                dt_object = datetime.utcfromtimestamp(raw_value) + timedelta(hours=9)
                return dt_object.strftime('%Y/%m/%d %H:%M:%S')

        # Milestone
        elif type_id == 9:
            if not self.milestone:
                self.milestone = self._send_get(f'get_milestones/{self.project_id}', 'milestones')
            return self._find_name_by_id(self.milestone, 'name', raw_value)

        # BDD Scenario
        elif type_id == 13:
            value = ''
            raw_value_json = json.loads(raw_value)
            for elem in raw_value_json:
                value += elem['content'] + '\n'
            return value

        # BDD Scenario Result
        elif type_id == 14:
            value = ''
            raw_value_json = json.loads(raw_value)
            for elem in raw_value_json:
                value += f"内容:\n{elem['content']}\n\n実際の結果:\n{elem['actual']}\n\nステータス:\n{self._get_actual_value('status_id', 0, 0, elem['status_id'])}\n\n"
            return value

        # その他（テキスト）
        else:
            return raw_value

class WebError(Exception):
    pass

#---------------------------------------------------------------------------------
# main

instruction = """

 ### TestRail テストランCSVエクスポート ###

  テストランに登録されたテスト結果をCSVファイルとして出力します。

  使い方：
    > python  tr_csv_export.py  ID  OUTPUT_CSV_FILENAME MODE
    * ID: 
      * マイルストーン：対象のマイルストーンをWebブラウザで開いた際に表示される index.php?/milestones/view/xxx のxxxの数字）
      * テスト計画： 対象のテスト計画をWebブラウザで開いた際に表示される index.php?/plans/view/xxx のxxxの数字）
      * テストラン： 対象のテストランをWebブラウザで開いた際に表示される index.php?/runs/view/xxx のxxxの数字）
    * OUTPUT_CSV_FILENAME: 出力するCSVファイル名
    * MODE: milestone, testplan, testrun のいずれか

  実行例：
    > python  tr_csv_export.py  18  testrun_18.csv testrun
    * テストランID が 18 のテスト結果を testrun_18.csv としてCSVファイルに出力します。

  事前準備：
    1. TestRail の [管理] > [サイト設定] > [API] タブで 'API の有効化' チェックボックスを有効化してください。
    2. スクリプト内の上部に記載されたTestRailのログイン情報をご利用環境に合わせて変更してください。

"""

def main():
    args = sys.argv
    if len(args) != 4 or args[3] not in ["milestone", "testplan", "testrun"]:
        print(instruction)
        exit(0)
    
    testrun_id = args[1]
    output_csv_file = args[2]
    mode = args[3]
    charcode = 'cp932'
    client = TestRailTestRunExporter(testrail_url, user, password, charcode)
    client.run(testrun_id, output_csv_file, mode)

if __name__ == "__main__":
    main()
